% function cost.m
%
% finding the total cost of the project
%
% by Darren Chang (darrenz@mit.edu)
%
% last modified: 8/12/2003

% This module calculates the total present cost
% of a satellite program mostly based on the
% procedure outlined in
% Larson and Wertz, Chapter 20.

% Steps 1: find Space Mission Characteristics.
% Some of these have been calculated or set up in previous modules.
% For those that have not been set up, I'll set them to
% arbitrary values here.
function totalCost = cost(N,NP,alti,AntW,CEW,SCBDW,SWW,BLP,FSKLOC,GSKLOC,...
    spaceLife,IDTime,NGS,personnel,AKM,AKMDW,AKMImpulse,DR,minLaunchCost,Risl,ISL,addDVCost,addFUCost)
N;						% # of sats in the constellation
alti;					% orbit altitude, in km
PLType = 'Comm';	% payload type is always Comm
AntW;					% antenna weight, in kg
CEW;					% comm electonics weight, in kg
SCBDW;				% spacecraft bus dry weight, in kg
SWW;					% satellite wet weight, in kg
BLP;					% beginning of life power, in W
FSKLOC;				% flight software lines of code, in thousand lines
GSKLOC;				% ground software lines of code, in thousand lines
spaceLife;			% space segment design life, in years
IDTime;				% initial deployment time, in years
NGS;					% number of gateways
personnel;			% number of operations and support personnel
spareSC = NP;		% number of spare spacecraft = number of planes in the constellation
AKM;					% type of apogee kick motor.  0 for no AKM, 1 for spin-stablized and 2 for 3-axis-stablized
AKMDW;				% AKM dry weight
AKMImpulse;			% total impulse of AKM
minLaunchCost;		% from LV.m
DR;					% discount rate in percentage

% Step 2: find RDT&E Hardware Cost in FY2002$K
% Assuming Nominal new design with some heritage
% Assuming commercial program, the cost factored by 0.8
% All money values converted from 1992 to 2002 USD by multiplying 1.369

Cant		= 1015*AntW^0.59;		% cost of antenna
CcommEle	= 917*CEW^0.7;			% cost comm electronics
CscBus	= 16253 + 110*SCBDW;	% cost of spacecraft bus
if AKM == 1							% cost of apogee kick motor, 1 for spin-stabilized   									
   Cakm = 490 + 0.0518*AKMImpulse;	
elseif AKM == 2					% 2 for 3-axis stablized
   Cakm = 0.0156*AKMImpulse;
else
   Cakm = 0;
end;
RDTEHC = (Cant + CcommEle + CscBus + Cakm + addDVCost)*1.369; % RDT&E Hardware Cost

% Step 3: find TFU Hardware Cost in FY2002$K
% Assuming commercial program
% All money values converted from 1992 to 2002 USD by multiplying 1.369

Cant		= 20 + 230*AntW^0.59;
CcommEle	= 179*CEW;
CscBus	= 185*SCBDW^0.77;
if AKM == 1							
   Cakm = 58*AKMDW^0.72;	
elseif AKM == 2
   Cakm = 0.0052*AKMImpulse;
else
   Cakm = 0;
end;
TFUHC = (Cant + CcommEle + CscBus + Cakm + addFUCost)*1.369; % TFU Hardware Cost

% Step 4: find hardware cost of every unit in FY2002$K
% first, find learning curve slope LCS in percentage
% Assume that for each plane there is a spare satellite.
totN = N + spareSC;		% total number of sats
if totN <= 10
   LCS = 95;
elseif 10 < totN & totN <=50
   LCS = 90;
else
   LCS = 85;
end;
% second, find the hardware cost of each unit
B = 1 - log(100/LCS)/log(2);
previousCost = 0;
for i = 1:totN;
   LCF = i^B;	% learning curve factor
   UHC(i) = TFUHC*LCF - previousCost;
   previousCost = TFUHC*LCF;
end;

% Step 5: find Aerospace Ground Equipment Cost for RDT&E in FY2002$K
AGEC = 0.11*(RDTEHC + TFUHC);

% Step 6: find total Program Level Cost for RDT&E, TFU, and all the rest of the units in FY2002$K
RDTEPLC = 0.36*RDTEHC;
UPLC = 0.33.*UHC;
totPLC  = RDTEPLC + sum(UPLC);

% Step 7: find Launch Operations and Orbital Support Cost for all units in FY2002$K
if AKM == 0
   LOOSC = 2.51*SWW;
else
   LOOSC = 64 + 1.44*SWW;
end;
totLOOSC = LOOSC*totN*1.369;

% Step 8: find Flight Software Cost in FY2002$K
FSC = FSKLOC*375*1.369;

% Step 9: find Launch Vehicle Cost in FY2002$K
LVC = minLaunchCost;

% Step 10: find Ground Software Cost in FY2002$K
% Assuming using UNIX - C
GSC = GSKLOC*190*1.67*1.369;

% Step 11: find total Ground Segment Development Cost in FY2002$K
GSDC = (18 + 81 + 100 + 15 + 18 + 30 + 15 + 24)/100*GSC;
% command center cost, assuming two CC built, cost based on GlobalStar, ARIES, ORBCOMM, and Starnet
CCC = 11856*1.369;
totGSDC = NGS*GSDC + 2*CCC;

% Step 12: find Initial Deployment Cost in FY2002$K
IDC = RDTEHC + sum(UHC) + AGEC + totPLC + totLOOSC + FSC + totGSDC + LVC;

% Step 13: find IDC distribution over Initial Deployment Time
% Assume 60% of expenditure at schedule midpoint
previousCostFraction = 0;
% If more than two sats are included, 50% expenditure at schedule midpoint is suggested by SMAD
% and therefore the A and B values.
Atmp = 0;
Btmp = 1;
for i = 1:IDTime
   Stmp = i/IDTime;
   costFraction = Atmp*(10 + Stmp*((15 - 4*Stmp)*Stmp - 20))*Stmp^2 + Btmp*(10 + Stmp*(6*Stmp - 15))*Stmp^3 +...
       (1 - (Atmp + Btmp))*(5 - 4*Stmp)*Stmp^4;
   CostPerYear(i) = (costFraction - previousCostFraction)*IDC;
   previousCostFraction = costFraction;
end;

% Step 14: find Operation and Support Cost per year during Space Segment Life Time in FY2002$K
% Assume contractor labor
maintCost = (18 + 81 + 100)/100*GSC*0.1;
staff = 300;
laborCost = 140*(personnel + staff);
%laborCost = 140*(20*3 + staff);
OSCPerYear = maintCost*NGS + laborCost;

% Step 15: find OSC distribution over Space Segment Life Time
for i = (IDTime + 1):(IDTime + spaceLife)
   CostPerYear(i) = OSCPerYear;
end;

% Step 16: discount Cost Per Year to Present Cost
%for i = 1:(IDTime + spaceLife)
%   PCPerYear(i) = CostPerYear(i)/(1+DR/100)^i;
%end;
%PC = sum(PCPerYear);

% Step 16: add 10% Fee
%totalPresentCost = PC*(1 + 0.1);
%totalPresentCost = totalPresentCost*2;

totalCost = sum(CostPerYear);
totalCost = totalCost*(1 + 0.1);